//==============================================================================
//
// Title:		XTDR_DLL_Test.cpp
// Purpose:		XTDR DLL (v2.6.0) test program for MS Visual C++
//
// Copyright:	HYPERLABS INC. All Rights Reserved.
//
//==============================================================================


// Includes
#include "stdafx.h"
#include <iostream>
#include <string>

// Include DLLs (FTDI and ZTDR)
#include "XTDR_2XX.h"
#pragma comment(lib, "XTDR_2XX.lib")


int _tmain(int argc, _TCHAR* argv[])
{
	// Variable to store returns from DLL calls
	int status = 0;

	// Set window limits based on X unit (m, ft, ns)
	int xLimit[3] = { 5, 15, 75 }; // Maximum window width
	double xMin[3] = { 0.2, 0.6, 3.0 }; // Minimum window width

	// Dummy variable values; setting any of these manually will disable the prompts below
	int x = -1;
	int y = -1;
	double start = -1.0;
	double end = -1.0;
	double k = -1.0;
	int isDiff = -1;

	// Shorthand for unit selection
	std::string xName[3] = { "m", "ft", "ns" };

	// Initialize FTDI device
	std::cout << "XTDR DLL Test Program (C++) | Copyright 2016 HYPERLABS INC." << std::endl << std::endl;
	std::cout << "Initializing TDR...";
	status = xtdr_init();

	// Finish startup message
	if (status == 1)
	{
		// Initialization OK, continue
		std::cout << " DONE!" << std::endl;
	}
	else
	{
		// Alert to failure and exit program
		std::cout << " FAILED!"<< status << std::endl << std::endl;

		return 0;
	}

	// Calibrate active channel	
	std::cout << "Calibrating TDR...";
	status = xtdr_cal(2,4,1);

	if (status == 1)
	{
		// Initialization OK, continue
		std::cout << " DONE!" << std::endl;
	}
	else
	{
		// Alert to failure and exit program
		std::cout << " FAILED!" << status << std::endl << std::endl;

		return 0;
	}

	// Set X units (m, ft, ns)
	while (x != 0 && x != 1 && x != 2)
	{
		std::cout << std::endl;
		std::cout << "Enter X units: m [0], ft [1], ns [2]" << std::endl;

		if (!(std::cin >> x))
		{
			std::cin.clear(); //clear the failure flag if there is an error when reading!
		}
	}

	// Set Y units (mV, Norm, Ohm, Rho)
	while (y != 0 && y != 1 && y != 2 && y != 3)
	{
		std::cout << std::endl;
		std::cout << "Enter Y units: mV [0], Norm [1], Ohm [2], Rho [3]" << std::endl;

		if (!(std::cin >> y))
		{
			std::cin.clear(); //clear the failure flag if there is an error when reading!
		}
	}

	// Set acquisition window start (max depends on unit)
	while (start < 0 || start > 75)
	{
		std::cout << std::endl;
		std::cout << "Enter acquisition window start (0 to " << xLimit[x] << " " << xName[x] << "):" << std::endl;

		if (!(std::cin >> start))
		{
			std::cin.clear(); //clear the failure flag if there is an error when reading!
		}
	}

	// Set acquisition window end (at least a certain distance beyond start, but not past width limit)
	while (end < 0 || end <= start + xMin[x] || end > start + xLimit[x])
	{
		std::cout << std::endl;
		std::cout << "Enter acquisition window end (" << start + xMin[x] << " to " << start + xLimit[x] << " " << xName[x] << "):" << std::endl;

		if (!(std::cin >> end))
		{
			std::cin.clear(); //clear the failure flag if there is an error when reading!
		}
	}

	// Set dielectric K
	while (k <= 0.0)
	{
		std::cout << std::endl;
		std::cout << "Enter dielectric K (2.25 for standard coax):" << std::endl;

		if (!(std::cin >> k))
		{
			std::cin.clear(); //clear the failure flag if there is an error when reading!
		}
	}

	// Send environmental variables to device
	status = xtdr_environ(x, y, start, end, k);

	// Set horizontal reference to open
	std::cout << std::endl;
	std::cout << "Setting horizontal reference. Make sure active channel is in open." << std::endl;
	std::cout << std::endl;

	system("pause");

	// Set zero on single-ended line
	status = xtdr_zero(-1.0, 0);

	// Differential (Ohm only) or single-ended acquisition
	if (y == 2)
	{
		while (isDiff < 0 || isDiff > 1)
		{
			std::cout << std::endl;
			std::cout << "Acquire differential waveform: NO [0], YES [1]:" << std::endl;

			if (!(std::cin >> isDiff))
			{
				std::cin.clear(); //clear the failure flag if there is an error when reading!
			}
		}
	}

	// Acquire and average n waveforms
	int n = 0;

	while (n < 1 || n > 16)
	{
		std::cout << std::endl;
		std::cout << "Enter number of waveforms to average (1-16):" << std::endl;

		if (!(std::cin >> n))
		{
			std::cin.clear(); //clear the failure flag if there is an error when reading!
		}
	}

	// Acquire the actual waveform data from device
	if (isDiff == 1)
	{
		// Differential waveform
		status = xtdr_diff(n);
	}
	else
	{
		// Single-ended waveform
		status = xtdr_acquire(n);
	}
	
	// Store waveforms to local arrays of size 1024 (one for each data point)
	double wfmDataX[1024];
	double wfmDataY[1024];

	// In this example, these data are not used; arrays populated only to show functionality
	int i = 0;

	for (i = 0; i < 1024; i++)
	{
		wfmDataX[i] = xtdr_getX(i);
		wfmDataY[i] = xtdr_getY(i);
	}

	// Write CSV file to disk
	char filename[128];
	std::cout << std::endl;
	std::cout << "Enter filename to store (include .csv extension)" << std::endl;

	if (!(std::cin >> filename))
	{
		std::cin.clear(); //clear the failure flag if there is an error when reading!
	}

	// Dump data to selected CSV file
	status = xtdr_dump(filename);

	std::cout << std::endl;
	std::cout << "Test program complete. Look for CSV in program directory." << std::endl;
	std::cout << std::endl;
	
	system("pause");

	return 0;
}