//==============================================================================
//
// Title:		ZTDR_DLL_Test.cpp
// Purpose:		ZTDR DLL (v1.1.0) test program for MS Visual C++
//
// Copyright:	HYPERLABS. All Rights Reserved.
//
//==============================================================================

// Includes
#include "stdafx.h"
#include <iostream>
#include <string>

// Include DLLs (FTDI and ZTDR)
#include "ZTDR_1XX.h"
#pragma comment(lib, "ZTDR_1XX.lib")

int _tmain(int argc, _TCHAR* argv[])
{
	// Variable to store returns from DLL calls
	int status = 0;

	// Set window limits based on X unit (m, ft, ns)
	int xLimit[3] = { 400, 1200, 2000 }; // Maximum window width
	int xMin[3] = { 2, 10, 6 }; // Minimum window width
	
	// Shorthand for unit selection
	std::string xName[3] = { "m", "ft", "ns" };

	// Startup message
	std::cout << "ZTDR DLL Test Program (C++) | Copyright 2014 HYPERLABS INC." << std::endl << std::endl;
	std::cout << "Initializing TDR...";

	// Initialize and calibrate TDR
	status = initDevice();

	// Finish startup message
	if (status == 1)
	{
		// Initialization OK, continue
		std::cout << " DONE!" << std::endl;
	}
	else
	{
		// Alert to failure and exit program
		std::cout << " FAILED!" << std::endl << std::endl;

		return 0;
	}

	// Acquisition enviornment variable definitions
	int x = -1;
	int y = -1;
	double start = -1.0;
	double end = -1.0;
	double k = -1.0;

	// In v1.0.1 of DLL, acquisition fixed at 1024 data points
	const int rec = 1024;
	
	// Set X units (m, ft, ns)
	while (x != 0 && x != 1 && x != 2) 
	{
		std::cout << std::endl;
		std::cout << "Enter X units: m [0], ft [1], ns [2]" << std::endl;
		
		if (!(std::cin >> x))
		{
			std::cin.clear(); //clear the failure flag if there is an error when reading!
		}
	}

	// Set Y units (mV, Norm, Ohm, Rho)
	while (y != 0 && y != 1 && y != 2 && y != 3)
	{
		std::cout << std::endl;
		std::cout << "Enter Y units: mV [0], Norm [1], Ohm [2], Rho [3]" << std::endl;
		
		if (!(std::cin >> y))
		{
			std::cin.clear(); //clear the failure flag if there is an error when reading!
		}
	}

	// Set acquisition window start (max depends on unit)
	while (start < 0 || start > 400)
	{
		std::cout << std::endl;
		std::cout << "Enter acquisition window start (0 to " << xLimit[x] << " " << xName[x] << "):" << std::endl;
		
		if (!(std::cin >> start))
		{
			std::cin.clear(); //clear the failure flag if there is an error when reading!
		}
	}

	// Set acquisition window end (at least a certain distance beyond start, but not past width limit)
	while (end < 0 || end <= start + xMin[x] || end > start + xLimit[x])
	{
		std::cout << std::endl;
		std::cout << "Enter acquisition window end (" << start + xMin[x] << " to " << start + xLimit[x] << " " << xName[x] << "):" << std::endl;
		
		if (!(std::cin >> end))
		{
			std::cin.clear(); //clear the failure flag if there is an error when reading!
		}
	}


	// Set dielectric K
	while (k <= 0.0)
	{
		std::cout << std::endl;
		std::cout << "Enter dielectric K (2.25 for standard coax):" << std::endl;

		if (!(std::cin >> k))
		{
			std::cin.clear(); //clear the failure flag if there is an error when reading!
		}
	}

	// Send environmental variables to device
	// In v1.0.1, required that rec = 1024
	status = setEnviron(x, y, start, end, k, 1024);
	
	// Perform vertical calibration
	status = vertCal();

	// Set horizontal reference to open
	std::cout << std::endl;
	std::cout << "Setting horizontal reference. Make sure instrument is in open and hit ENTER." << std::endl;

	status = setRefX(-1.0);

	// Acquire and average n waveforms
	int n = 0;
	
	while (n < 1 || n > 16)
	{
		std::cout << std::endl;
		std::cout << "Enter number of waveforms to average (1-16):" << std::endl;

		if (!(std::cin >> n))
		{
			std::cin.clear(); //clear the failure flag if there is an error when reading!
		}
	}

	// Acquire the actual waveform data from device
	status = acquireWaveform(n);

	// Store waveforms to local arrays of size 1024 (one for each data point)
	double wfmDataX[1024];
	double wfmDataY[1024];

	// In this example, these data are not used; arrays populated only to show functionality
	int i = 0;

	for (i = 0; i < 1024; i++)
	{
		wfmDataX[i] = fetchDataX(i);
		wfmDataY[i] = fetchDataY(i);
	}

	// Write CSV file to disk
	char filename[128];
	std::cout << std::endl;
	std::cout << "Enter filename to store (include .csv extension)" << std::endl;

	if (!(std::cin >> filename))
	{
		std::cin.clear(); //clear the failure flag if there is an error when reading!
	}

	// Dump data to selected CSV file
	status = dumpFile(filename);

	std::cout << std::endl;
	std::cout << "Test program complete. Look for CSV in program directory." << std::endl;
	std::cout << std::endl;

	return 0;
}